// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateEmployeeType,
  deleteEmployeeType,
  viewEmployeeType,
} from "../controller/employeeTypeController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
/**
 * @swagger
 * tags:
 *   name: Employee Type
 *   description: API operations for managing all Employee Type
 */
const employeeTypeRoutes = Router();

/**
 * @swagger
 * /api/employeeType/create-update-employee-type :
 *   post:
 *     summary: Create or update an Employee Type
 *     tags: [Employee Type]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the employee type "
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Employee Type created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
employeeTypeRoutes.post(
  "/create-update-employee-type",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      342,
      "employee_type"
    )(req, res, next);
  },
  controllerWrapper(createUpdateEmployeeType)
);

/**
 * @swagger
 * /api/employeeType/get-employee-type :
 *   get:
 *     summary: Get list of all EmployeeType
 *     tags: [Employee Type]
 *     responses:
 *       200:
 *         description: List of employeeType
 *       500:
 *         description: Internal server error
 */
employeeTypeRoutes.get(
  "/get-employee-type",
  verifyToken,
  sideBarPermissionCheck("view", 342, "employee_type"),
  controllerWrapper(viewEmployeeType)
);

/**
 * @swagger
 * /api/employeeType/get-employee-type /{id}:
 *   get:
 *     summary: Get a specific record of Employee Type
 *     tags: [Employee Type]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the employee-type
 *     responses:
 *       200:
 *         description: Employee Type data
 *       404:
 *         description: Employee Type not found
 *       500:
 *         description: Internal server error
 */
employeeTypeRoutes.get(
  "/get-employee-type/:id",
  verifyToken,
  sideBarPermissionCheck("view", 342, "employee_type"),
  controllerWrapper(viewEmployeeType)
);

/**
 * @swagger
 * /api/employeeType/delete-employee-type/{id}:
 *   delete:
 *     summary: Delete a specific record of Employee Type
 *     tags: [Employee Type]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the employee-type  to delete
 *     responses:
 *       200:
 *         description: Employee Type successfully deleted
 *       404:
 *         description: Employee Type not found
 *       500:
 *         description: Internal server error
 */
employeeTypeRoutes.delete(
  "/delete-employee-type/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 342, "employee_type"),
  controllerWrapper(deleteEmployeeType)
);

export default employeeTypeRoutes;
