// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deleteFireEquipmentRegister,
  FireEquipmentRegisterCreateUpdate,
  updateFireEquipmentRegisterHistoryStatus,
  updateFireEquipmentRegisterStatus,
  viewFireEquipmentRegister,
  viewFireEquipmentRegisterStatusHistory,
} from "../controller/fireEquipmentRegisterController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { decodeSuperAdminToken } from "../helper/verifyTokenSuperAdmin.js";

/**
 * @swagger
 * tags:
 *   name: FireEquipmentRegister
 *   description: API Operations For Managing Fire Equipment Register
 */
const FireEquipmentRegisterRoutes = Router();

/**
 * @swagger
 * /api/fireEquipmentRegister/create-update-fire-equipment-register:
 *   post:
 *     summary: Create Or Update A Fire Equipment Register
 *     tags: [FireEquipmentRegister]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               equipment_type:
 *                 type: integer
 *                 example: 1
 *               equipment_name:
 *                 type: string
 *                 example: Fire Extinguisher
 *               location:
 *                 type: integer
 *                 example: 9
 *               unique_equipment_id:
 *                 type: integer
 *                 example: 3434
 *               installation_date:
 *                 type: string
 *                 format: date
 *                 example: 2020-10-20
 *               supplier_name:
 *                 type: string
 *                 example: Manufacturer
 *               modal_number:
 *                 type: string
 *                 example: 45
 *               capacity:
 *                 type: string
 *                 example: 1
 *               department:
 *                 type: string
 *                 example:
 *               status:
 *                 type: string
 *                 example: Maintenance Overdue
 *               organization:
 *                 type: integer
 *                 example: 14
 */
FireEquipmentRegisterRoutes.post(
  "/create-update-fire-equipment-register",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 301, "fire_equipment_register")(req, res, next);
  },
  controllerWrapper(FireEquipmentRegisterCreateUpdate)
);

/**
 * @swagger
 * /api/fireEquipmentRegister/get-fire-equipment-register:
 *   get:
 *     summary: Get list of all Fire Equipment Registers
 *     tags: [FireEquipmentRegister]
 */
FireEquipmentRegisterRoutes.get(
  "/get-fire-equipment-register",
  verifyToken,
  sideBarPermissionCheck("view", 301, "fire_equipment_register"),
  controllerWrapper(viewFireEquipmentRegister)
);

/**
 * @swagger
 * /api/fireEquipmentRegister/get-fire-equipment-register/{id}:
 *   get:
 *     summary: Get A Specific Fire Equipment Register record
 *     tags: [FireEquipmentRegister]
 */
FireEquipmentRegisterRoutes.get(
  "/get-fire-equipment-register/:id",
  verifyToken,
  sideBarPermissionCheck("view", 301, "fire_equipment_register"),
  controllerWrapper(viewFireEquipmentRegister)
);

/**
 * @swagger
 * /api/fireEquipmentRegister/delete-fire-equipment-register/{id}:
 *   delete:
 *     summary: Delete a specific record of Fire Equipment Register
 *     tags: [FireEquipmentRegister]
 */
FireEquipmentRegisterRoutes.delete(
  "/delete-fire-equipment-register/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 301, "fire_equipment_register"),
  controllerWrapper(deleteFireEquipmentRegister)
);

FireEquipmentRegisterRoutes.post(
  "/update-fire-equipment-register-status",
  verifyToken,
  sideBarPermissionCheck("edit", 301, "fire_equipment_register"),
  controllerWrapper(updateFireEquipmentRegisterStatus)
);

FireEquipmentRegisterRoutes.get(
  "/get-fire-equipment-register-status-history",
  verifyToken,
  // decodeSuperAdminToken,
  sideBarPermissionCheck("view", 301, "fire_equipment_register"),
  controllerWrapper(viewFireEquipmentRegisterStatusHistory)
);

FireEquipmentRegisterRoutes.get(
  "/get-fire-equipment-register-status-history/:id",
  verifyToken,
  sideBarPermissionCheck("view", 301, "fire_equipment_register"),
  controllerWrapper(viewFireEquipmentRegisterStatusHistory)
);

FireEquipmentRegisterRoutes.post(
  "/update-fire-equipment-register-history-status",
  verifyToken,
  // sideBarPermissionCheck("edit", 313, "fire_equipment_register"),
  controllerWrapper(updateFireEquipmentRegisterHistoryStatus)
);

export default FireEquipmentRegisterRoutes;
