import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createInfrastructureTechnicalForm,
  getInfrastructureTechnicalForm,
  deleteInfrastructureTechnicalForm,
} from "../controller/infrastructureTechnicalFormController.js";

const infrastructureTechnicalFormRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Infrastructure Technical Forms
 *   description: API operations for managing infrastructure technical forms
 */

/**
 * @swagger
 * /api/infrastructureTechnicalForm/create-infrastructure-technical-form:
 *   post:
 *     summary: Create a new infrastructure technical form entry
 *     tags: [Infrastructure Technical Forms]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               category_id:
 *                 type: integer
 *               incident_id:
 *                 type: integer
 *               date_of_occurrence:
 *                 type: string
 *                 format: date
 *               type_of_property_asset_damaged:
 *                 type: string
 *               specify_property_asset_damaged:
 *                 type: string
 *               property_damaged_details:
 *                 type: string
 *               select_property_damaged:
 *                 type: string
 *               case_reported_to_insurer:
 *                 type: integer
 *               name_of_insurer:
 *                 type: string
 *               date_reported_to_insurer:
 *                 type: string
 *                 format: date
 *               reported_by_whom_insurer:
 *                 type: string
 *               claim_no_insurer:
 *                 type: string
 *               case_reported_to_saps:
 *                 type: integer
 *               name_of_police_station:
 *                 type: string
 *               case_no_saps:
 *                 type: string
 *               name_of_contact_person:
 *                 type: string
 *               contact_details:
 *                 type: string
 *               severity_of_damage:
 *                 type: string
 *                 enum: [Minor, Major, Beyond Repair]
 *               estimated_cost_of_damage:
 *                 type: number
 *                 format: float
 *               comments:
 *                 type: string
 *               form_completed_by:
 *                 type: string
 *               designation:
 *                 type: string
 *               signature:
 *                 type: string
 *               date:
 *                 type: string
 *                 format: date
 *               department:
 *                 type: integer
 *               organization:
 *                 type: integer
 *           example:
 *             category_id: 1
 *             incident_id: 1001
 *             date_of_occurrence: "2023-07-12"
 *             type_of_property_asset_damaged: "Building"
 *             specify_property_asset_damaged: "Office Wall"
 *             property_damaged_details: "Crack on the wall"
 *             select_property_damaged: "Yes"
 *             case_reported_to_insurer: 1
 *             name_of_insurer: "Insurer Inc."
 *             date_reported_to_insurer: "2023-07-13"
 *             reported_by_whom_insurer: "John Doe"
 *             claim_no_insurer: "CLM123456"
 *             case_reported_to_saps: 0
 *             name_of_police_station: "Station Name"
 *             case_no_saps: "C123"
 *             name_of_contact_person: "Jane Doe"
 *             contact_details: "jane@example.com"
 *             severity_of_damage: "Major"
 *             estimated_cost_of_damage: 2500.75
 *             comments: "Urgent repair needed"
 *             form_completed_by: "Alice"
 *             designation: "Safety Officer"
 *             signature: "Alice Signature"
 *             date: "2023-07-12"
 *             department: 10
 *             organization: 5
 *       responses:
 *         '200':
 *           description: Successfully created a new infrastructure technical form entry
 *         '400':
 *           description: Bad request. Invalid input parameters
 *         '500':
 *           description: Internal server error
 */
infrastructureTechnicalFormRoutes.post(
  "/create-infrastructure-technical-form",
  verifyToken,
  // sideBarPermissionCheck(req.body.id ? "edit" : "create", 176, "safety_incident_technical_form"),
  controllerWrapper(createInfrastructureTechnicalForm)
);

/**
 * @swagger
 * /api/infrastructureTechnicalForm/get-infrastructure-technical-form/{id}:
 *   get:
 *     summary: Get an infrastructure technical form entry by ID
 *     tags: [Infrastructure Technical Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the infrastructure technical form entry to get
 *     responses:
 *       '200':
 *         description: Successfully retrieved the infrastructure technical form entry
 *       '404':
 *         description: Infrastructure technical form entry not found
 *       '500':
 *         description: Internal server error
 */
infrastructureTechnicalFormRoutes.get(
  "/get-infrastructure-technical-form/:id",
  verifyToken,
  // sideBarPermissionCheck("view", 176, "safety_incident_technical_form"),
  controllerWrapper(getInfrastructureTechnicalForm)
);

/**
 * @swagger
 * /api/infrastructureTechnicalForm/delete-infrastructure-technical-form/{id}:
 *   delete:
 *     summary: Delete an infrastructure technical form entry by ID
 *     tags: [Infrastructure Technical Forms]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the infrastructure technical form entry to delete
 *     responses:
 *       '200':
 *         description: Successfully deleted the infrastructure technical form entry
 *       '404':
 *         description: Infrastructure technical form entry not found
 *       '500':
 *         description: Internal server error
 */
infrastructureTechnicalFormRoutes.delete(
  "/delete-infrastructure-technical-form/:id",
  verifyToken,
  // sideBarPermissionCheck("delete", 176, "safety_incident_technical_form"),
  controllerWrapper(deleteInfrastructureTechnicalForm)
);

export default infrastructureTechnicalFormRoutes;
