// Import necessary modules
import { Router } from "express";
import {
  createUpdateMeeting,
  viewAllMeeting,
  deleteMeeting,
  viewAllMeetingCalendar,
} from "../controller/meetingController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { meetingCreateUpdateSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Meeting
 *   description: API operations for managing all Meeting
 */
const meetingRoutes = Router();

/**
 * @swagger
 * /api/meeting/create-update-meeting:
 *   post:
 *     summary: Create or update a Meeting
 *     tags: [Meeting]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: {
 *             "id": 1,
 *             "meeting_title": "Weekly Sync",
 *             "meeting_hierarchy": "Department Meeting",
 *             "participants": ["1", "2"],
 *             "meeting_owner": "Owner Name",
 *             "planned_meeting_date_from": "2024-01-01T10:00:00Z",
 *             "planned_meeting_date_to": "2024-01-01T11:00:00Z",
 *             "venue": "Conference Room A",
 *             "agenda": "Discuss project updates",
 *             "organization": "Org Name",
 *             "department": "Department Name"
 *           }
 *     responses:
 *       200:
 *         description: Meeting created or updated successfully
 *       400:
 *         description: Bad request due to validation errors
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingRoutes.post(
  "/create-update-meeting",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 23, "meeting")(
      req,
      res,
      next
    );
  },
  validateRequest(meetingCreateUpdateSchema),
  controllerWrapper(createUpdateMeeting)
);

/**
 * @swagger
 * /api/meeting/get-meeting:
 *   get:
 *     summary: Get all list of Meeting
 *     tags: [Meeting]
 *     responses:
 *       200:
 *         description: A list of all meetings
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingRoutes.get(
  "/get-meeting",
  verifyToken,
  sideBarPermissionCheck("view", 23, "meeting"),
  controllerWrapper(viewAllMeeting)
);

/**
 * @swagger
 * /api/meeting/get-meeting/{id}:
 *   get:
 *     summary: Get a specific record of Meeting
 *     tags: [Meeting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting
 *     responses:
 *       200:
 *         description: A specific meeting record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Meeting not found
 *       500:
 *         description: Internal server error
 */
meetingRoutes.get(
  "/get-meeting/:id",
  verifyToken,
  sideBarPermissionCheck("view", 23, "meeting"),
  controllerWrapper(viewAllMeeting)
);

/**
 * @swagger
 * /api/meeting/delete-meeting/{id}:
 *   delete:
 *     summary: Delete a specific record of Meeting
 *     tags: [Meeting]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting
 *     responses:
 *       200:
 *         description: Successfully deleted
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Meeting not found
 *       500:
 *         description: Internal server error
 */
meetingRoutes.delete(
  "/delete-meeting/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 23, "meeting"),
  controllerWrapper(deleteMeeting)
);

/**
 * @swagger
 * /api/meeting/get-meeting-calendar:
 *   post:
 *     summary: Get a meeting calendar download
 *     tags: [Meeting]
 *     responses:
 *       200:
 *         description: Meeting calendar retrieved successfully
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingRoutes.post(
  "/get-meeting-calendar",
  verifyToken,
  sideBarPermissionCheck("view", 23, "meeting"),
  controllerWrapper(viewAllMeetingCalendar)
);

export default meetingRoutes;
