// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateOperationalRiskIdentification,
  getOperationalRiskIdentification,
  deleteOperationalRiskIdentification,
  approveRejectOperationalRiskIdentification,
  getOperationalRiskIdentificationApprovalWorkflow,
  updateOperationalRiskIdentification,
} from "../controller/operationalRiskIdentificationController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { OperationalRiskIdentificationUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { validateRequest } from "../helper/general.js";

/**
 * @swagger
 * tags:
 *   name: Operational Risk Identification
 *   description: API Operations For Managing All Operational Risk Identifications
 */
const operationalRiskIdentificationRoutes = Router();

/**
 * @swagger
 * /api/operationalRiskIdentification/create-update-operational-risk-identification:
 *   post:
 *     summary: Create Or Update An Operational Risk Identification
 *     tags: [Operational Risk Identification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               risk_name:
 *                 type: string
 *               risk_register_id:
 *                 type: integer
 *               work_area:
 *                 type: string
 *               main_process:
 *                 type: string
 *               activity_service:
 *                 type: string
 *               activity_frequency:
 *                 type: string
 *               risk_owner:
 *                 type: string
 *               hazard_name:
 *                 type: string
 *               hazard_description:
 *                 type: string
 *               category_description:
 *                 type: string
 *               risk_description:
 *                 type: string
 *               impact_description:
 *                 type: string
 *               frequency:
 *                 type: integer
 *               probability:
 *                 type: integer
 *               consequence:
 *                 type: integer
 *               risk_rating:
 *                 type: string
 *               risk_ranking:
 *                 type: integer
 *               control_id:
 *                 type: integer
 *               control_name:
 *                 type: string
 *               control_description:
 *                 type: string
 *               control_effectiveness:
 *                 type: string
 *               residual_risk_rating:
 *                 type: string
 *               residual_risk_ranking:
 *                 type: integer
 *               opportunity_identification:
 *                 type: string
 *               opportunity_description:
 *                 type: string
 *               priority_confirmation:
 *                 type: string
 *               organization:
 *                 type: string
 *               department:
 *                 type: string
 *               substitution:
 *                 type: string
 *               engineering_control:
 *                 type: string
 *               administrative_control:
 *                 type: string
 *               last_resort:
 *                 type: string
 *           example:
 *             id: 1
 *             risk_name: "Risk of Fire"
 *             risk_register_id: 101
 *             work_area: "Warehouse"
 *             main_process: "Storage"
 *             activity_service: "Goods Storage"
 *             activity_frequency: "Daily"
 *             risk_owner: "John Doe"
 *             hazard_name: "Flammable Materials"
 *             hazard_description: "Presence of flammable materials."
 *             category_description: "Fire Safety"
 *             risk_description: "Potential fire outbreak."
 *             impact_description: "High impact on safety."
 *             frequency: 5
 *             probability: 4
 *             consequence: 3
 *             risk_rating: "High"
 *             risk_ranking: 1
 *             control_id: 201
 *             control_name: "Fire Extinguishers"
 *             control_description: "Availability of fire extinguishers."
 *             control_effectiveness: "Effective"
 *             residual_risk_rating: "Medium"
 *             residual_risk_ranking: 2
 *             opportunity_identification: "Safety Training"
 *             opportunity_description: "Training staff on fire safety."
 *             priority_confirmation: "High"
 *             organization: "Company A"
 *             department: "Safety"
 *             substitution: "Use non-flammable materials."
 *             engineering_control: "Sprinkler System"
 *             administrative_control: "Regular inspections"
 *             last_resort: "Emergency drills"
 */

operationalRiskIdentificationRoutes.post(
  "/create-update-operational-risk-identification",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      38,
      "operational_risk_identification"
    )(req, res, next);
  },
  validateRequest(OperationalRiskIdentificationUpdateValidationSchema),
  controllerWrapper(createUpdateOperationalRiskIdentification)
);

/**
 * @swagger
 * /api/operationalRiskIdentification/get-operational-risk-identification:
 *   get:
 *     summary: Get All Operational Risk Identification List
 *     tags: [Operational Risk Identification]
 */
operationalRiskIdentificationRoutes.get(
  "/get-operational-risk-identification",
  verifyToken,
  sideBarPermissionCheck("view", 38, "operational_risk_identification"),
  controllerWrapper(getOperationalRiskIdentification)
);

/**
 * @swagger
 * /api/operationalRiskIdentification/update-tactical-risk-identification-priority:
 *   post:
 *     summary: Update Tactical Risk Identification Priority
 *     tags: [Operational Risk Identification]
 */
operationalRiskIdentificationRoutes.post(
  "/update-operational-risk-identification-priority",
  verifyToken,
  sideBarPermissionCheck("edit", 38, "operational_risk_identification"),
  controllerWrapper(updateOperationalRiskIdentification)
);

/**
 * @swagger
 * /api/operationalRiskIdentification/get-operational-risk-identification:
 *   get:
 *     summary: Get All Operational Risk Identification List
 *     tags: [Operational Risk Identification]
 */
operationalRiskIdentificationRoutes.get(
  "/get-operational-risk-identification-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 38, "operational_risk_identification"),
  controllerWrapper(getOperationalRiskIdentificationApprovalWorkflow)
);

/**
 * @swagger
 * /api/operationalRiskIdentification/get-operational-risk-identification/{id}:
 *   get:
 *     summary: Get A Specific Operational Risk Identification Record
 *     tags: [Operational Risk Identification]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The operational risk identification ID
 */
operationalRiskIdentificationRoutes.get(
  "/get-operational-risk-identification/:id",
  verifyToken,
  sideBarPermissionCheck("view", 38, "operational_risk_identification"),
  controllerWrapper(getOperationalRiskIdentification)
);

/**
 * @swagger
 * /api/operationalRiskIdentification/delete-operational-risk-identification/{id}:
 *   delete:
 *     summary: Delete a specific record of operational risk identification
 *     tags: [Operational Risk Identification]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The operational risk identification ID
 */
operationalRiskIdentificationRoutes.delete(
  "/delete-operational-risk-identification/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 38, "operational_risk_identification"),
  controllerWrapper(deleteOperationalRiskIdentification)
);

/**
 * @swagger
 * /api/operationalRiskIdentification/approve-reject-operational-risk-identification:
 *   post:
 *     summary: Approve or reject operational risk identification
 *     tags: [Operational Risk Identification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               operationalRiskIdentificationId:
 *                 type: string
 *               status:
 *                 type: string
 *               comment:
 *                 type: string
 *           example:
 *             operationalRiskIdentificationId: "12345"
 *             status: "approved"
 *             comment: "Approved after review."
 */
operationalRiskIdentificationRoutes.post(
  "/approve-reject-operational-risk-identification",
  verifyToken,
  sideBarPermissionCheck("view", 38, "operational_risk_identification"),
  controllerWrapper(approveRejectOperationalRiskIdentification)
);

export default operationalRiskIdentificationRoutes;
