import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  assignPpeRepository,
  createUpdatePpeRepository,
  deletePpeRepository,
  getAssignedPpeRepository,
  getPpeRepository,
  getPpeRepositoryQuantity,
} from "../controller/ppeRepositoryController.js";

/**
 * @swagger
 * tags:
 *   name: PpeRepository
 *   description: API operations for managing PpeRepository
 */
const ppeRepositoryRoutes = Router();

/**
 * @swagger
 * /api/ppeRepository/create-update-ppe-repository:
 *   post:
 *     summary: Create or update a PpeRepository
 *     tags: [PpeRepository]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *                 description: The ID of the PpeRepository record (for update).
 *               effective_date:
 *                 type: string
 *                 format: date
 *                 description: The effective date of the PPE item.
 *               last_inspection_date:
 *                 type: string
 *                 format: date
 *                 description: The date of the last inspection.
 *               location:
 *                 type: string
 *                 description: The location where the PPE is stored.
 *               next_inspection_date:
 *                 type: string
 *                 format: date
 *                 description: The date of the next inspection.
 *               organization:
 *                 type: integer
 *                 description: The organization ID associated with the PPE.
 *               quantity:
 *                 type: integer
 *                 description: The quantity of the PPE items.
 *               review_date:
 *                 type: string
 *                 format: date
 *                 description: The review date for the PPE.
 *               status:
 *                 type: string
 *                 description: The status of the PPE item.
 *               type:
 *                 type: string
 *                 description: The type of PPE.
 *           example:
 *             {
 *               id: 1,
 *               effective_date: "2024-10-25",
 *               last_inspection_date: "2024-10-25",
 *               location: "Delhi",
 *               next_inspection_date: "2024-10-26",
 *               organization: 18,
 *               quantity: 10,
 *               review_date: "2024-10-25",
 *               status: "Sufficient",
 *               type: "Type 1"
 *             }
 */
ppeRepositoryRoutes.post(
  "/create-update-ppe-repository",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 309, "ppe_repository")(req, res, next);
  },
  controllerWrapper(createUpdatePpeRepository)
);

/**
 * @swagger
 * /api/ppeRepository/get-ppe-repository:
 *   get:
 *     summary: Get all records of PpeRepository
 *     tags: [PpeRepository]
 */

ppeRepositoryRoutes.get(
  "/get-ppe-repository",
  verifyToken,
  sideBarPermissionCheck("view", 309, "ppe_repository"),
  controllerWrapper(getPpeRepository)
);

/**
 * @swagger
 * /api/ppeRepository/get-ppe-repository/{id}:
 *   get:
 *     summary: Get a specific record of PpeRepository
 *     tags: [PpeRepository]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         description: The ID of the PpeRepository record to retrieve.
 *         schema:
 *           type: integer
 */
ppeRepositoryRoutes.get(
  "/get-ppe-repository/:id",
  verifyToken,
  sideBarPermissionCheck("view", 309, "ppe_repository"),
  controllerWrapper(getPpeRepository)
);

ppeRepositoryRoutes.get(
  "/get-ppe-repository-quantity",
  verifyToken,
  sideBarPermissionCheck("view", 309, "ppe_repository"),
  controllerWrapper(getPpeRepositoryQuantity)
);

/**
 * @swagger
 * /api/ppeRepository/delete-ppe-repository/{id}:
 *   delete:
 *     summary: Delete a specific record of PpeRepository
 *     tags: [PpeRepository]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         description: The ID of the PpeRepository record to delete.
 *         schema:
 *           type: integer
 */
ppeRepositoryRoutes.delete(
  "/delete-ppe-repository/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 309, "ppe_repository"),
  controllerWrapper(deletePpeRepository)
);
ppeRepositoryRoutes.post("/assign-ppe-repository", verifyToken, controllerWrapper(assignPpeRepository));

ppeRepositoryRoutes.get("/get-assigned-repository", verifyToken, controllerWrapper(getAssignedPpeRepository));
export default ppeRepositoryRoutes;
