import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateStakeholderClassification,
  getStakeholderClassification,
  deleteStakeholderClassification,
} from "../controller/stakeholderClassificationController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const stakeholderClassificationRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Stakeholder Classification
 *   description: API operations for managing stakeholder classification
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     StakeHolderClassification:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated id of the stakeholder classification
 *         stakeholder_type:
 *           type: integer
 *           description: Type of stakeholder
 *         stakeholder_category:
 *           type: string
 *           description: Category of stakeholder
 *         stakeholder_department_internal:
 *           type: integer
 *           description: Internal department of stakeholder
 *         stakeholder_name_internal:
 *           type: integer
 *           description: Internal name of stakeholder
 *         stakeholder_name_external:
 *           type: string
 *           description: External name of stakeholder
 *         role_external:
 *           type: string
 *           description: External role of stakeholder
 *         email:
 *           type: string
 *           description: Email of stakeholder
 *         phone:
 *           type: string
 *           description: Phone number of stakeholder
 *         address:
 *           type: string
 *           description: Address of stakeholder
 *         classification:
 *           type: string
 *           description: Classification of stakeholder
 *         engagement_level:
 *           type: string
 *           description: Engagement level of stakeholder
 *         status:
 *           type: string
 *           description: Status of stakeholder
 *         relationship_type:
 *           type: string
 *           description: Relationship type of stakeholder
 *         interaction_frequency:
 *           type: string
 *           description: Interaction frequency of stakeholder
 *         influence_level:
 *           type: string
 *           description: Influence level of stakeholder
 *         connection_strength:
 *           type: string
 *           description: Connection strength of stakeholder
 *         interest_score:
 *           type: string
 *           description: Interest score of stakeholder
 *         influence_score:
 *           type: string
 *           description: Influence score of stakeholder
 *         assessment_date:
 *           type: string
 *           format: date
 *           description: Assessment date of stakeholder
 *         priority_level:
 *           type: string
 *           description: Priority level of stakeholder
 *         assessment_notes:
 *           type: string
 *           description: Assessment notes of stakeholder
 *         organization:
 *           type: integer
 *           description: Organization of stakeholder
 *         department:
 *           type: integer
 *           description: Department of stakeholder
 *         deleted:
 *           type: integer
 *           description: Soft delete flag
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: Creation timestamp
 *         created_by:
 *           type: integer
 *           description: Creator of the stakeholder classification
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: Last update timestamp
 *         updated_by:
 *           type: integer
 *           description: Last updater of the stakeholder classification
 *       example:
 *         id: 1
 *         stakeholder_type: 1
 *         stakeholder_category: "Internal"
 *         stakeholder_department_internal: 10
 *         stakeholder_name_internal: 101
 *         stakeholder_name_external: "John Doe"
 *         role_external: "Manager"
 *         email: "john.doe@example.com"
 *         phone: "1234567890"
 *         address: "123 Main St"
 *         classification: "High"
 *         engagement_level: "Medium"
 *         status: "Active"
 *         relationship_type: "Partner"
 *         interaction_frequency: "Weekly"
 *         influence_level: "High"
 *         connection_strength: "Strong"
 *         interest_score: "High"
 *         influence_score: "Medium"
 *         assessment_date: "2024-07-22"
 *         priority_level: "High"
 *         assessment_notes: "Key stakeholder in the project"
 *         organization: 1
 *         department: 2
 *         deleted: 0
 *         created_at: "2024-07-22T14:00:00Z"
 *         created_by: 1
 *         updated_at: "2024-07-22T14:00:00Z"
 *         updated_by: 1
 */

/**
 * @swagger
 * api/stakeholderClassification/create-update-stakeholder-classification:
 *   post:
 *     summary: Create or update a stakeholder classification
 *     tags: [Stakeholder Classification]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: 1
 *             stakeholder_type: 1
 *             stakeholder_category: "Internal"
 *             stakeholder_department_internal: 10
 *             stakeholder_name_internal: 101
 *             stakeholder_name_external: "John Doe"
 *             role_external: "Manager"
 *             email: "john.doe@mail.com"
 *             phone: "1234567890"
 *             address: "123 Main St"
 *             classification: "High"
 *             engagement_level: "Medium"
 *             status: "Active"
 *             relationship_type: "Partner"
 *             interaction_frequency: "Weekly"
 *             influence_level: "High"
 *             connection_strength: "Strong"
 *             interest_score: "High"
 *             influence_score: "Medium"
 *             assessment_date: "2024-07-22"
 *             priority_level: "High"
 *             assessment_notes: "Key stakeholder in the project"
 *             organization: 1
 *             department: 2
 *             deleted: 0
 *             created_at: "2024-07-22T14:00:00Z"
 *             created_by: 1
 *             updated_at: "2024-07-22T14:00:00Z"
 *             updated_by: 1
 *     responses:
 *       200:
 *         description: The created/updated stakeholder classification
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeHolderClassification'
 */
stakeholderClassificationRoutes.post(
  "/create-update-stakeholder-classification",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      225,
      "stakeholder_classification"
    )(req, res, next);
  },
  controllerWrapper(createUpdateStakeholderClassification)
);

/**
 * @swagger
 * api/stakeholderClassification/get-stakeholder-classification:
 *   get:
 *     summary: Get all stakeholder classifications
 *     tags: [Stakeholder Classification]
 *     responses:
 *       200:
 *         description: List of all stakeholder classifications
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/StakeHolderClassification'
 */
stakeholderClassificationRoutes.get(
  "/get-stakeholder-classification",
  verifyToken,
  sideBarPermissionCheck("view", 225, "stakeholder_classification"),
  controllerWrapper(getStakeholderClassification)
);

/**
 * @swagger
 * api/stakeholderClassification/get-stakeholder-classification/{id}:
 *   get:
 *     summary: Get a stakeholder classification by id
 *     tags: [Stakeholder Classification]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The stakeholder classification id
 *     responses:
 *       200:
 *         description: The stakeholder classification by id
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeHolderClassification'
 */
stakeholderClassificationRoutes.get(
  "/get-stakeholder-classification/:id",
  verifyToken,
  sideBarPermissionCheck("view", "225", "stakeholder_classification"),
  controllerWrapper(getStakeholderClassification)
);

/**
 * @swagger
 * api/stakeholderClassification/delete-stakeholder-classification/{id}:
 *   delete:
 *     summary: Delete a stakeholder classification by id
 *     tags: [Stakeholder Classification]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The stakeholder classification id
 *     responses:
 *       200:
 *         description: Stakeholder classification deleted
 */
stakeholderClassificationRoutes.delete(
  "/delete-stakeholder-classification/:id",
  verifyToken,
  sideBarPermissionCheck("delete", "225", "stakeholder_classification"),
  controllerWrapper(deleteStakeholderClassification)
);

export default stakeholderClassificationRoutes;
