import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createUpdateStakeholderEngagementPlan,
  getStakeholderEngagementPlan,
  deleteStakeholderEngagementPlan,
  getActivityStructureOnFrequency,
} from "../controller/stakeholderEngagementPlanController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { stakeholderEngagementPlanValidationSchema } from "../validationsRecord/validationSchema.js";
import { validateRequest } from "../helper/general.js";

const stakeholderEngagementPlanRoutes = Router();

stakeholderEngagementPlanRoutes.post(
  "/create-update-stakeholder-engagement-plan",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      227,
      "stakeholder_engagement_plan"
    )(req, res, next);
  },
  // validateRequest(stakeholderEngagementPlanValidationSchema),
  controllerWrapper(createUpdateStakeholderEngagementPlan)
);

stakeholderEngagementPlanRoutes.get(
  "/get-stakeholder-engagement-plan",
  verifyToken,
  sideBarPermissionCheck("view", "227", "stakeholder_engagement_plan"),
  controllerWrapper(getStakeholderEngagementPlan)
);

stakeholderEngagementPlanRoutes.get(
  "/get-stakeholder-engagement-plan/:id",
  verifyToken,
  sideBarPermissionCheck("view", "227", "stakeholder_engagement_plan"),
  controllerWrapper(getStakeholderEngagementPlan)
);

stakeholderEngagementPlanRoutes.delete(
  "/delete-stakeholder-engagement-plan/:id",
  verifyToken,
  sideBarPermissionCheck("delete", "227", "stakeholder_engagement_plan"),
  controllerWrapper(deleteStakeholderEngagementPlan)
);

stakeholderEngagementPlanRoutes.get(
  "/get-plan-activity-structure/:id",
  verifyToken,
  controllerWrapper(getActivityStructureOnFrequency)
);

export default stakeholderEngagementPlanRoutes;

/**
 * @swagger
 * tags:
 *   name: Stakeholder Engagement Plan
 *   description: API operations for managing stakeholder engagement plans
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     StakeholderEngagementPlan:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the stakeholder engagement plan
 *         classification_type:
 *           type: string
 *           description: Type of classification for the engagement
 *         priority_level:
 *           type: string
 *           description: Priority level of the engagement plan
 *         stakeholder_category:
 *           type: integer
 *           description: Category ID of the stakeholder
 *         stakeholder_name:
 *           type: string
 *           description: Name of the stakeholder
 *         recommended_engagement_level:
 *           type: string
 *           description: Recommended engagement level for the stakeholder
 *         engagement_plan_name:
 *           type: string
 *           description: Name of the engagement plan
 *         engagement_plan_objectives:
 *           type: string
 *           description: Objectives of the engagement plan
 *         engagement_frequency:
 *           type: string
 *           description: Frequency of engagement, e.g., "Bi-weekly"
 *         engagement_start_date:
 *           type: string
 *           format: date
 *           description: Start date of the engagement
 *         no_of_cycles:
 *           type: string
 *           description: Number of engagement cycles
 *         engagement_methods:
 *           type: integer
 *           description: Engagement method ID
 *         engagement_level:
 *           type: integer
 *           description: Level of engagement
 *         identification_id:
 *           type: integer
 *           description: ID for stakeholder identification
 *         responsible_person:
 *           type: integer
 *           description: ID of the responsible person for the engagement
 *         human_resource_requirement:
 *           type: string
 *           description: JSON string detailing human resource requirements
 *         technological_resources:
 *           type: string
 *           description: JSON string detailing technological resources required
 *         type_of_resources:
 *           type: string
 *           description: Type of resources needed for engagement
 *         amount:
 *           type: integer
 *           format: int64
 *           description: Budgeted amount for engagement resources
 *         currency:
 *           type: integer
 *           description: Currency ID for the budget amount
 *       example:
 *         id: 1
 *         classification_type: "High Priority"
 *         priority_level: "High"
 *         stakeholder_category: 3
 *         stakeholder_name: "Smriti"
 *         recommended_engagement_level: "Engage proactively"
 *         engagement_plan_name: "Community Outreach Plan"
 *         engagement_plan_objectives: "Increase community involvement"
 *         engagement_frequency: "Bi-weekly"
 *         engagement_start_date: "2024-11-10"
 *         no_of_cycles: "2"
 *         engagement_methods: 9
 *         engagement_level: 4
 *         identification_id: 5
 *         responsible_person: 34
 *         human_resource_requirement: '[{"role":18,"role_name":"Test Role","no_of_peoples":34}]'
 *         technological_resources: '[{"resources":"Resource 12"},{"resources":"Resource 2"}]'
 *         type_of_resources: "Technology Investments"
 *         amount: 4000
 *         currency: 7
 */

/**
 * @swagger
 * /api/stakeholderEngagementPlan/create-update-stakeholder-engagement-plan:
 *   post:
 *     summary: Create or update a stakeholder engagement plan
 *     tags: [Stakeholder Engagement Plan]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/StakeholderEngagementPlan'
 *     responses:
 *       200:
 *         description: The created or updated stakeholder engagement plan
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeholderEngagementPlan'
 */

/**
 * @swagger
 * /api/stakeholderEngagementPlan/get-stakeholder-engagement-plan:
 *   get:
 *     summary: Get all stakeholder engagement plans
 *     tags: [Stakeholder Engagement Plan]
 *     responses:
 *       200:
 *         description: List of all stakeholder engagement plans
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/StakeholderEngagementPlan'
 */

/**
 * @swagger
 * /api/stakeholderEngagementPlan/get-stakeholder-engagement-plan/{id}:
 *   get:
 *     summary: Get a stakeholder engagement plan by ID
 *     tags: [Stakeholder Engagement Plan]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The stakeholder engagement plan ID
 *     responses:
 *       200:
 *         description: The stakeholder engagement plan by ID
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeholderEngagementPlan'
 */

/**
 * @swagger
 * /api/stakeholderEngagementPlan/delete-stakeholder-engagement-plan/{id}:
 *   delete:
 *     summary: Delete a stakeholder engagement plan by ID
 *     tags: [Stakeholder Engagement Plan]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: The stakeholder engagement plan ID
 *     responses:
 *       200:
 *         description: Stakeholder engagement plan deleted
 */



/**
 * @swagger
 * /api/stakeholderEngagementPlan/get-plan-activity-structure/:id:
 *   post:
 *     summary: Get plan activity structure
 *     tags: [Stakeholder Engagement Plan]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/StakeholderEngagementPlan'
 *     responses:
 *       200:
 *         description: The get plan activity structure on the basis of frequency of engagement
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/StakeholderEngagementPlan'
 */