// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  deleteStrategicRisk,
  strategicRiskCreateUpdate,
  viewStrategicRisk,
} from "../controller/strategicRiskController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { strategicRiskValidationSchema } from "../validationsRecord/validationSchema.js";
import { strategicRiskAssessmentExcel } from "../helper/strategicRiskAssessmentExcel.js";

/**
 * @swagger
 * tags:
 *   name: StrategicRisk
 *   description: API Operations For Managing strategic risk
 */
const strategicRiskRoutes = Router();

/**
 * @swagger
 * /api/strategicRisk/create-update-strategic-risk:
 *   post:
 *     summary: Create Or Update A strategic risk
 *     tags: [StrategicRisk]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: "id"
 *             risk_register_name: "risk_register_name"
 *             assessment_date: "assessment_date"
 *             review_date: "review_date"
 *             risk_context: "risk_context"
 *             risk_assessment_facilitator: "risk_assessment_facilitator"
 *             risk_register_approver: "risk_register_approver"
 *             meeting_id: "meeting_id"
 *             organization: "organization"
 *             department: "department"
 */
strategicRiskRoutes.post(
  "/create-update-strategic-risk",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      32,
      "strategic_risk"
    )(req, res, next);
  },
  validateRequest(strategicRiskValidationSchema),
  controllerWrapper(strategicRiskCreateUpdate)
);

/**
 * @swagger
 * /api/strategicRisk/get-strategic-risk:
 *   get:
 *     summary: Get list of all strategic risks
 *     tags: [StrategicRisk]
 */
strategicRiskRoutes.get(
  "/get-strategic-risk",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk"),
  controllerWrapper(viewStrategicRisk)
);

/**
 * @swagger
 * /api/strategicRisk/get-strategic-risk/{id}:
 *   get:
 *     summary: Get A Specific strategic risk record
 *     tags: [StrategicRisk]
 */
strategicRiskRoutes.get(
  "/get-strategic-risk/:id",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk"),
  controllerWrapper(viewStrategicRisk)
);

/**
 * @swagger
 * /api/strategicRisk/delete-strategic-risk/{id}:
 *   delete:
 *     summary: Delete a specific record of strategic risk
 *     tags: [StrategicRisk]
 */
strategicRiskRoutes.delete(
  "/delete-strategic-risk/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 32, "strategic_risk"),
  controllerWrapper(deleteStrategicRisk)
);

// /**
//  * @swagger
//  * /api/strategicRisk/approve-strategic-risk:
//  *   post:
//  *     summary: Approve A Specific strategic risk
//  *     tags: [StrategicRisk]
//  */
// strategicRiskRoutes.post("/approve-strategic-risk", verifyToken, sideBarPermissionCheck('edit', 32, 'strategic_risk'), approveStrategicRisk);

strategicRiskRoutes.get(
  "/get-strategic-risk-heat-map-excel",
  verifyToken,
  sideBarPermissionCheck("view", 32, "strategic_risk"),
  controllerWrapper(strategicRiskAssessmentExcel)
);


export default strategicRiskRoutes;
